# Copyright 2021-2022 MinusOne, Inc.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation
# files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,
# modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom
# the Software is furnished to do so, subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR
# IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
import uuid
import sys
import json
import random
import time
import string
import requests
from datetime import datetime
from datetime import timedelta
import math

from m1 import M1Client

CONFIG_SERVER_KEY   = "server"
CONFIG_STORE_KEY    = "session"
CONFIG_USERNAME_KEY = "username"
CONFIG_PASSWORD_KEY = "password"
CONFIG_IP_KEY       = "ip"

HEARTBEAT_INTERVAL = 5

# This is a very simple tool to generate synthetic events.

def prettyTimestamp(epoch):
	return datetime.fromtimestamp(epoch/1000 if epoch else time.time()).strftime("%Y-%m-%dT%H:%M:%S")

def randomDate(d1, d2):
	return d1 + timedelta(days = random.randint(0, (d2 - d1).days))

def randomElement(l):
	return l[random.randint(0, len(l) - 1)]

def sendSessionStore(event):
	if CONFIG.get(CONFIG_STORE_KEY):
		CLIENT.postRetryForever("/session/update", { "store" : CONFIG.get(CONFIG_STORE_KEY),	"ops" : json.dumps([{"upsert" : [event]}])})
	else:
		sendWrite([event])
	
def sendWrite(events):
	if events:
		print(f"sending {len(events)} interaction events")
		CLIENT.postRetryForever("/write", {"items" : json.dumps(events), "publish" : True })

def makeEvent(eventType, user, session, ip):
	return {
		"_m1" : {"ip" : ip},
		"id" : str(uuid.uuid4()),
		"time": int(time.time()) * 1000,
		"session": session,
		"user" : user,
		"event": {
			"type":eventType
		},
		"page": {
			"domain": "www." + randomWord() + ".com",
			"hash": randomWord(),
			"port": 100,
			"scrollTop": random.randint(1, 1000),
			"timeOnPage": 100,
		},
		"referrer": {
			"port": 100,
		},
		"screen": {
			"angle": random.randint(0, 360),
			"availableHeight": random.randint(100, 2000),
			"availableWidth": random.randint(100, 3840),
			"colorDepth": 2 ** random.randint(1, 8),
			"height": random.randint(100, 2000),
			"orientation": random.choice(["portrait", "landscape"]),
			"pixelDepth": 2 ** random.randint(1, 8),
			"width": random.randint(100, 3840)
		},

		"settings": {
			"cookies": False,
			"doNotTrack": False,
		},
		"system": {
			"os": random.choice(["windows", "android", "ios", "linux", "macos"])
		},
		"window": {
			"height": 3840,
			"width":  2000
		},
	}


def randomWord():
	return WORDS[random.randint(0, len(WORDS) - 1)]	

with open("words") as f:
	WORDS = [x.strip() for x in f if x.strip()]

def writeTestFile(N, ip):
	with open(f"{N}.json", "w+") as f:
		query = [{"upsert" : [makeEvent(str(uuid.uuid4()), str(uuid.uuid4())) for i in range(N)]}]
		json.dump(query, f, indent = 2)

def makeUser():
	return {		
		"id":str(uuid.uuid4()),
		"displayName":(randomWord() + " " + randomWord()).title(),
		"gender":random.choice(["male", "female"]),
		"dateOfBirth":int((randomDate(datetime(1925, 1, 1), datetime.now()) - datetime(1970, 1, 1)).total_seconds()),
	}

def makeSession():
	return {
		"id": str(uuid.uuid4()),
		"start": int(time.time() * 1000)
	}

def getUserEvents(user, session, ip):
	events = []
	if random.random() <= .1:  
		events.append(makeEvent("userDidSomething", user, session, ip))
	return events

def simulate():
	time.sleep(random.random() * 3)

	while True:
		user = makeUser()
		session = makeSession()
		ip = random.choice(["73.4.243.88", # Boston
												"38.68.135.125", # Dallas
												"128.218.229.26", # SF
												"103.211.55.244", # Delhi
												"2600:8804:1085:600:e803:e577:545a:51c6", # Omaha
												"72.200.1.74", # New Orleans
												"64.150.5.135" # Chicago
												])
		watchTime = random.randint(5000, 5 * 60 * 1000) # range of 5s to 5min; want to see some churn even if that's not necessarily realistic
		print(f"{user['displayName']} logs in; session {session['id']} begins at {prettyTimestamp(session['start'])}")
		
		sendWrite([makeEvent("userLogin", user, None, ip),
							 makeEvent("sessionStart", user, session, ip)])
						
		while True:
			print(f"{user['displayName']} still here")
			sendSessionStore(makeEvent("heartbeat", user, session, ip))
			time.sleep(HEARTBEAT_INTERVAL)
			sendWrite(getUserEvents(user, session, ip))		
			if time.time() * 1000 - session["start"] > watchTime:
				print(f"{user['displayName']} went away around {prettyTimestamp(time.time() * 1000)}")
				sendWrite([makeEvent("sessionEnd", user, session, ip)])
				time.sleep(random.random() * 1)
				break


with open("default.json" if len(sys.argv) == 1 else sys.argv[1]) as f:
	CONFIG = json.load(f)
	CLIENT = M1Client(CONFIG[CONFIG_SERVER_KEY], CONFIG[CONFIG_USERNAME_KEY], CONFIG[CONFIG_PASSWORD_KEY], CONFIG.get(CONFIG_IP_KEY))

simulate()
