package m1.util;

import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;

public abstract class Query {
	public static final int MAX_TERMS = 100_000;

	public static String or(Map<String, Object> m) {
		return conjunct(m, " OR ");
	}
	public static String and(Map<String, Object> m) {
		return conjunct(m, " AND ");
	}
	public static String clause(String property, Object values) {
		if(values instanceof Collection) {
			Collection<String> nonNullValues = ((Collection) values).stream().filter(v -> v != null).toList();
			return nonNullValues.isEmpty() ? null : valuesTerm(property, nonNullValues);
		} else {
			return values == null ? null : String.format("%s:%s", property, queryValue((String) values));
		}
	}

	public static String valuesTerm(String property, Collection<String> values) {
		return "(" + Transform.page(values, MAX_TERMS)
			.stream()
			.map(valuesPage -> String.format("%s:(%s)", property, orValues(valuesPage)))
			.collect(Collectors.joining(" OR ")) + ")";
	}

	private static String orValues(Collection<String> values) {
		return values
			.stream()
			.map(v -> queryValue(v))
			.collect(Collectors.joining("||"));
	}

	protected static final Set<Character> CHARS_TO_ESCAPE = new TreeSet<>() {
		{
			add('+');
			add('&');
			add('|');
			add('!');
			add('(');
			add(')');
			add('[');
			add(']');
			add('{');
			add('}');
			add('^');
			add('"');
			add('~');
			add('*');
			add('?');
			add(':');
			add('/');
			add('\\');
		}
	};

	public static String queryValue(String val) {
		if(val == null) {
			return null;
		}
		StringBuilder sb = new StringBuilder();
		sb.append('"');
		for(int i = 0; i < val.length(); i++) {
			if(CHARS_TO_ESCAPE.contains(val.charAt(i))) {
				sb.append('\\');
			}
			if('+' == val.charAt(i)) {
				sb.append("%2B");
			} else if('%' == val.charAt(i)) {
				sb.append("%25");
			} else if('&' == val.charAt(i)) {
				sb.append("%26");
			} else {
				sb.append(val.charAt(i));
			}
		}
		sb.append('"');
		return sb.toString();
	}

	private static String conjunct(Map<String, Object> m, String conjunction) {
		return m
			.entrySet()
			.stream()
			.map(es -> clause(es.getKey(), es.getValue()))
			.filter(clause -> clause != null)
			.collect(Collectors.joining(conjunction));
	}
}
