/********************************************************************************************************************************
 * Copyright 2021-2022 MinusOne, Inc.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom
 * the Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR
 * IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 ********************************************************************************************************************************/
package m1.util;

import java.net.URI;
import java.net.UnknownHostException;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.TimeUnit;
import org.apache.hc.client5.http.auth.AuthenticationException;
import org.apache.hc.client5.http.classic.methods.HttpGet;
import org.apache.hc.client5.http.classic.methods.HttpPost;
import org.apache.hc.client5.http.config.RequestConfig;
import org.apache.hc.client5.http.entity.GzipCompressingEntity;
import org.apache.hc.client5.http.entity.UrlEncodedFormEntity;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.client5.http.impl.classic.CloseableHttpResponse;
import org.apache.hc.client5.http.impl.classic.HttpClientBuilder;
import org.apache.hc.core5.http.ClassicHttpRequest;
import org.apache.hc.core5.http.HttpEntity;
import org.apache.hc.core5.http.NameValuePair;
import org.apache.hc.core5.http.io.entity.EntityUtils;
import org.apache.hc.core5.http.message.BasicNameValuePair;
import org.apache.hc.core5.net.URIBuilder;
import org.apache.hc.core5.util.Timeout;
import com.fasterxml.jackson.databind.ObjectMapper;
import m1.M1Response;

public abstract class Http {
	public static final String GET = "GET";
	public static final String POST = "POST";
	protected static final long INTERVAL_RETRY = 30L;

	////////////////////////////////////////////////////////////
	/// Public interface

	public static String request(String method, String url, Map<String, String> headers, Map params) throws Exception {
		try (M1Response response = requestStream(method, url, headers, params)) {
			return response.getString();
		}
	}
	public static M1Response requestStream(String method, String url, Map<String, String> headers, Map params) throws Exception {
		ClassicHttpRequest req;
		switch (method) {
			case POST:
				req = new HttpPost(url);
				HttpEntity payload = form(params);
				if(shouldCompressPayload(headers)) {
					payload = new GzipCompressingEntity(form(params));
				}
				req.setEntity(payload);

				break;
			default:
				req = new HttpGet(urlWithParams(url, params));
				break;
		}

		return http(req, headers);
	}
	protected static URI urlWithParams(String url, Map<String, String> params) throws Exception {
		URIBuilder out = new URIBuilder(url);
		if(params != null) {
			for(Entry<String, String> e : params.entrySet()) {
				out.setParameter(e.getKey(), e.getValue());
			}
		}
		return out.build();
	}

	////////////////////////////////////////////////////////////////////
	/// Protected Internals
	protected static boolean shouldCompressPayload(Map<String, String> headers) {
		if(headers != null) {
			for(String header : headers.keySet()) {
				if("Content-Encoding".equals(header) && "gzip".equals(headers.get(header))) {
					return true;
				}
			}
		}
		return false;
	}

	protected static M1Response http(ClassicHttpRequest req, Map<String, String> headers) throws Exception {
		if(headers != null) {
			for(Entry<String, String> e : headers.entrySet()) {
				req.addHeader(e.getKey(), e.getValue());
			}
		}
		RequestConfig rc = RequestConfig.custom().setResponseTimeout(Timeout.DISABLED).setConnectionRequestTimeout(Timeout.DISABLED).setConnectTimeout(Timeout.DISABLED).build();
		HttpClientBuilder builder = HttpClientBuilder.create().setDefaultRequestConfig(rc);
		try {
			CloseableHttpClient hc = builder.build();
			CloseableHttpResponse r = hc.execute(req);
			//System.out.println("URL: " + req.getRequestUri());

			HttpEntity e = r.getEntity();

			if(Arrays.asList(502, 503, 504).contains(r.getCode())) {
				TimeUnit.SECONDS.sleep(INTERVAL_RETRY);
				return http(req, headers);
			}
			if(r.getCode() == 401) {
				throw new AuthenticationException("Token invalid or incorrect login information.");
			}
			if(!Arrays.asList(200, 204).contains(r.getCode())) {
				throw new Exception((e == null ? "" : EntityUtils.toString(e)));
			}

			return new M1Response(r, hc);
		} catch(UnknownHostException un) {
			TimeUnit.SECONDS.sleep(INTERVAL_RETRY);
			return http(req, headers);
		}
	}

	protected static UrlEncodedFormEntity form(Map<String, Object> m) throws Exception {
		ObjectMapper mapper = new ObjectMapper();
		List<NameValuePair> params = new LinkedList<>();
		for(Entry<String, Object> e : m.entrySet()) {
			if(e.getValue() != null) {
				if(e.getValue() instanceof Collection || e.getValue() instanceof Map) {
					params.add(new BasicNameValuePair(e.getKey(), mapper.writeValueAsString(e.getValue())));
				} else {
					params.add(new BasicNameValuePair(e.getKey(), e.getValue().toString()));
				}
			}
		}
		return new UrlEncodedFormEntity(params);
	}
}
