/********************************************************************************************************************************
 * Copyright 2021-2022 MinusOne, Inc.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom
 * the Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR
 * IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 ********************************************************************************************************************************/
package m1;

import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import m1.util.Transform;

public class M1Ops {
	protected M1 m1;
	protected String opsName;
	public M1Ops() {
	}
	public M1Ops(String type) {
		this();
		m1 = new M1(type);
	}
	public M1Ops(M1 m1) {
		this(m1, M1.OPS);
	}
	public M1Ops(M1 m1, String opsName) {
		this();
		this.m1 = m1;
		this.opsName = opsName;
	}
	public void setM1(M1 m1) {
		this.m1 = m1;
	}
	public M1 getM1() {
		return m1;
	}

	// account creation
	public void signup() throws Exception {
		System.out.print("Username: ");
		String username = System.console().readLine();
		System.out.print("Password: ");
		String password = new String(System.console().readPassword());
		m1.POST(opsName, M1.url(opsName, "signup"), Map.of("username", username, "password", password));
		m1.auth(opsName, username, password);
	}

	// Create new environment via template file
	public void deploy(String envName, Map descriptor) throws Exception {
		// Create Environment
		System.out.println("Creating environment...");
		String e = createEnvironment(envName, descriptor);
		System.out.println(e);
		Map env = Transform.map(e);
		// Initialize Environment
		System.out.println("Initializing...");
		// Delay to avoid DNS negative caching timeout...
		TimeUnit.SECONDS.sleep(30);
		m1.init(envName, descriptor, env);
		deployStores(descriptor, envName);
		// Configure settings
		m1.waitForHealth(envName);
		System.out.println("Configuring...");
		m1.configure(envName, (List<Map<String, Map>>) Transform.getNestedProperty(descriptor, "environment", "configure"));
	}

	public String createEnvironment(String envName, Map descriptor) throws Exception {
		return m1.POST(opsName,
			M1.url(opsName, "env/create"),
			Transform.put((Map) Transform.getNestedProperty(descriptor, "environment", "create"), "env", envName));
	}

	public void deployStores(Map descriptor, String envName) throws Exception {
		createStores(envName, (List<Map>) Transform.getNestedProperty(descriptor, "environment", "stores"), "store");
		createStores(envName, (List<Map>) Transform.getNestedProperty(descriptor, "environment", "sessionStores"), "session");
	}
	public void createStores(String env, List<Map> stores, String type) throws Exception {
		for(Map store : stores) {
			store.put("env", env);
			createStore(env, type, store);
		}
	}
	public void createStore(String envName, String type, Map store) throws Exception {
		m1.POST(opsName, M1.url(opsName, "env/" + type + "/create"), store);
		m1.POST(envName, M1.url(envName) + "/" + type + "/add", store);
	}
	public void dropStore(String type, String env, String store) throws Exception {
		m1.POST(env, M1.url(env) + "/" + type + "/drop", Map.of("store", store));
		m1.POST(opsName, M1.url(opsName, "env/" + type + "/destroy"), Map.of("env", env, "store", store));
	}
	public void renameStore(String type, String env, String oldStoreName, String newStoreName) throws Exception {
		m1.POST(env, M1.url(env) + "/" + type + "/rename", Map.of("store", oldStoreName, "name", newStoreName));
		m1.POST(opsName, M1.url(opsName, "env/" + type + "/rename"), Map.of("env", env, "store", oldStoreName, "name", newStoreName));
	}
}
