/********************************************************************************************************************************
 * Copyright 2021-2022 MinusOne, Inc.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom
 * the Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR
 * IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 ********************************************************************************************************************************/
package m1;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Iterator;
import java.util.Map;
import m1.util.Transform;

public class M1DocumentIterator implements Iterator<Map>, AutoCloseable {
	protected final M1Response m1response;
	protected final BufferedReader reader;
	protected final InputStreamReader inputStreamReader;

	protected String currentLine;
	protected final Map responseMetadata;

	public M1DocumentIterator(M1Response m1response) throws Exception {
		this.m1response = m1response;
		this.inputStreamReader = new InputStreamReader(m1response.getInputStream(), StandardCharsets.UTF_8);
		this.reader = new BufferedReader(inputStreamReader);
		advance();
		responseMetadata = Transform.map(currentLine);
		advance();
	}

	public Map getResponseMetadata() {
		return responseMetadata;
	}

	public int getNumFound() {
		return (int) Transform.getNestedProperty(responseMetadata, "response", "numFound");
	}

	protected void advance() {
		try {
			this.currentLine = reader.readLine();
		} catch(IOException e) {
			e.printStackTrace();
			throw new RuntimeException(e);
		}
	}

	@Override
	public boolean hasNext() {
		return currentLine != null;
	}

	@Override
	public Map next() {
		try {
			Map toReturn = Transform.map(currentLine);
			advance();
			return toReturn;
		} catch(Exception e) {
			e.printStackTrace();
			throw new RuntimeException(e);
		}
	}

	@Override
	public void close() throws Exception {
		for(AutoCloseable closeable : Arrays.asList(reader, m1response, inputStreamReader)) {
			try {
				closeable.close();
			} catch(Exception swallowed) {
			}
		}
	}
}
