/********************************************************************************************************************************
 * Copyright 2021-2022 MinusOne, Inc.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom
 * the Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR
 * IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 ********************************************************************************************************************************/
package m1;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.TimeUnit;
import org.apache.hc.client5.http.auth.AuthenticationException;
import m1.util.Http;
import m1.util.Local;
import m1.util.Transform;

public class M1 {
	public static final String HEADER_AUTH = "m1-auth-token";
	public static final String M1_DOMAIN = ".minusonedb.com";
	public static final String LOCAL_OPS = "localops";
	public static final String LOCAL_OPS_URL = "http://localhost:8000";
	public static final String OPS = "ops";
	public static final String OPS_URL = "https://ops.minusonedb.com";

	protected static final String TYPE_LOCAL = "local";

	/// State and initializers
	private M1CredentialStore store;
	public M1() {
	}
	public M1(String type) {
		this();
		setType(type);
	}
	public void setType(String type) {
		if(TYPE_LOCAL.equals(type)) {
			this.store = new M1LocalCredentialStore();
		} else {
			this.store = new M1RAMCredentialStore();
		}
	}

	/// Public Interface 
	public String auth(String envName, String username, String password) throws Exception {
		String token = Http.request(Http.POST, M1.url(envName) + "/auth", null, Map.of("username", username, "password", password));
		if(store != null) {
			store.updateToken(envName, token, username, password);
		}
		return token;
	}
	protected void updateToken(String envName) throws Exception {
		Map cred = store.credential(envName);
		if(cred == null) {
			throw new Exception(String.format("No credentials present for %s; check for typos or use 'm1 auth %s' to store credentials.", envName, envName));
		}
		auth(envName, (String) cred.get(M1CredentialStore.USERNAME), (String) cred.get(M1CredentialStore.PASSWORD));
	}
	// Generic callers for standard services
	// NOTE: Both currently call resolveFiles on params, though this will (almost certainly) fail in the case of a
	// non-local implementation. Still, seems harmless: just don't use @...
	protected void call(String envName, Map<String, Map> svc) throws Exception {
		for(Entry<String, Map> call : svc.entrySet()) {
			POST(envName, M1.url(envName, call.getKey()), resolveFiles(call.getValue()));
		}
	}
	public String call(String envName, String method, String svc, Map params) throws Exception {
		try (M1Response m1response = request(method, envName, M1.url(envName, svc), resolveFiles(params))) {
			return m1response.getString();
		}
	}
	public String call(String envName, String method, String svc, Map params, Map<String, String> extraHeaders) throws Exception {
		try (M1Response m1response = request(method, envName, M1.url(envName, svc), resolveFiles(params), extraHeaders)) {
			return m1response.getString();
		}
	}
	protected void init(String envName, Map descriptor, Map env) throws Exception {
		String username = (String) Transform.getNestedProperty(descriptor, "environment", "init", "username");
		String password = (String) Transform.getNestedProperty(descriptor, "environment", "init", "password");
		String bucket = (String) Transform.getNestedProperty(env, "bucket");
		Http.request(Http.POST, M1.url(envName) + "/init", null, Map.of("username", username, "password", password, "bucket", bucket));
		// Time for db init to complete.
		TimeUnit.SECONDS.sleep(10);
		auth(envName, username, password);
	}
	public M1DocumentIterator getDocumentIterator(String envName, Map queryParams) throws Exception {
		queryParams = new HashMap(queryParams);
		queryParams.put("format", "jsonl");
		return new M1DocumentIterator(request(Http.POST, envName, M1.url(envName, "query"), queryParams));
	}

	/// Health checks and monitoring
	protected static final String HEALTH_OK = "PASSED";
	protected static final Long INTERVAL_WAIT = 30L;
	protected boolean health(String envName) throws Exception {
		String rsp = GET(envName, M1.url(envName) + "/health");
		Map health = Transform.map(rsp);
		return HEALTH_OK.equals(Transform.getNestedProperty(health, "overall", "status"));
	}
	protected void waitForHealth(String envName) throws Exception {
		TimeUnit.SECONDS.sleep(60);
		while(true) {
			if(!health(envName)) {
				TimeUnit.SECONDS.sleep(INTERVAL_WAIT);
			} else {
				return;
			}
		}
	}
	protected void configure(String envName, List<Map<String, Map>> ops) throws Exception {
		for(Map<String, Map> m : ops) {
			call(envName, m);
		}
	}

	// Http helpers
	public String GET(String envName, String url) throws Exception {
		try (M1Response m1response = request(Http.GET, envName, url, null)) {
			return m1response.getString();
		}
	}
	public String POST(String envName, String url, Map params) throws Exception {
		try (M1Response m1response = request(Http.POST, envName, url, params)) {
			return m1response.getString();
		}
	}
	public M1Response request(String method, String envName, String url, Map params) throws Exception {
		return request(method, envName, url, params, Collections.EMPTY_MAP);
	}
	public M1Response request(String method, String envName, String url, Map params, Map<String, String> extraHeaders) throws Exception {
		try {
			HashMap<String, String> headers = new HashMap<>();
			headers.put(M1.HEADER_AUTH, store.token(envName));
			for(String header : extraHeaders.keySet()) {
				headers.put(header, extraHeaders.get(header));
			}
			return Http.requestStream(method, url, headers, params);
		} catch(AuthenticationException ae) {
			updateToken(envName);
			return request(method, envName, url, params, extraHeaders);
		}
	}

	public static String envName(String envName) {
		return envName == null ? M1.OPS : envName;
	}
	public static String url(String envName) {
		if(envName == null || OPS.equals(envName)) {
			return OPS_URL;
		} else if("localdev".equals(envName)) {
			return "http://localhost:8100";
		} else if(LOCAL_OPS.equals(envName)) {
			return LOCAL_OPS_URL;
		} else {
			return "https://" + envName + M1_DOMAIN;
		}
	}
	public static String url(String envName, String svc) {
		return M1.url(envName) + "/" + (svc != null ? svc : "");
	}

	/// Protected internals
	protected static final String FILE_INDICATOR = "@";
	protected static Map resolveFiles(Map<String, Object> params) throws Exception {
		Map newParams = new HashMap();
		if(params != null) {
			for(String key : params.keySet()) {
				Object value = params.get(key);
				if(value instanceof String && ((String) value).startsWith(FILE_INDICATOR)) {
					value = Local.readFile(((String) value).substring(FILE_INDICATOR.length()));
				}
				newParams.put(key, value);
			}
		}
		return newParams;
	}
}
