/********************************************************************************************************************************
 * Copyright 2021-2022 MinusOne, Inc.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom
 * the Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR
 * IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 ********************************************************************************************************************************/
package m1;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import m1.util.Http;
import m1.util.Local;
import m1.util.Transform;

public class CLI {
	public static final List<String> ORCHESTRATE_CMDS = Arrays.asList("deploy", "addStores", "dropStore", "renameStore");

	public static void main(String[] args) throws Exception {
		// TODO: check number of args pre-emptively or add guard code.
		try {
			M1 m1 = new M1("local");
			if(args.length < 1) {
				System.out.println("Usage: m1 <cmd> <args...>");
				System.out.println("More help available at https://docs.minusonedb.com/#m1-client");
				return;
			}
			switch (args[0]) {
				case "auth":
					auth(m1, args.length > 1 && !"ops".equals(args[1]) ? args[1] : null);
					break;
				case "signup":
					new M1Ops(m1, M1.OPS).signup();
					break;
				case "orchestrate":
					if(ORCHESTRATE_CMDS.contains(args[1])) {
						doOrchestrate(new M1Ops(m1), Arrays.copyOfRange(args, 1, args.length));
					} else if(ORCHESTRATE_CMDS.contains(args[2])) {
						doOrchestrate(new M1Ops(m1, args[1]), Arrays.copyOfRange(args, 2, args.length));
					} else {
						System.out.println("orchestrate usage:");
						System.out.println("m1 orchestrate deploy <envName> <m1-template-file>");
						System.out.println("m1 orchestrate addStores <envName> <m1-template-file>");
						System.out.println("m1 orchestrate dropStore store|session <envName> <storeName>");
					}
					break;
				default:
					// Syntax: environment|ops svc args...
					String out = m1.call(args[0], method(args), args[1], params(Arrays.copyOfRange(args, 2, args.length)));
					System.out.println(out == null ? "" : out);
					break;
			}
		} catch(Exception ex) {
			System.out.println(ex.getMessage());
		}
	}
	// expectation is that first element of args is the orchestrate cmd and the rest are any command arguments
	protected static void doOrchestrate(M1Ops ops, String[] args) throws Exception {
		switch (args[0]) {
			case "deploy":
				Map descriptor = Transform.map(Local.readFile(args[2]));
				ops.deploy(args[1], descriptor);
				// Leave the user logged in.
				String username = (String) Transform.getNestedProperty(descriptor, "environment", "init", "username");
				String password = (String) Transform.getNestedProperty(descriptor, "environment", "init", "password");
				ops.getM1().auth(args[1], username, password);
				break;
			case "addStores":
				ops.deployStores(Transform.map(Local.readFile(args[2])), args[1]);
				break;
			case "renameStore":
				ops.renameStore(args[1], args[2], args[3], args[4]);
				break;
			case "dropStore":
				ops.dropStore(args[1], args[2], args[3]);
				break;
			default:
				throw new Exception(String.format("Unknown orchestrate command `%s`; must be one of deploy, addStores, or dropStore", args[0]));
		}
	}

	// User-prompting method appropriate for CLI
	protected static void auth(M1 m1, String envName) throws Exception {
		System.out.print("Username: ");
		String username = System.console().readLine();
		System.out.print("Password: ");
		String password = new String(System.console().readPassword());
		m1.auth(envName, username, password);
	}
	// Figure out HTTP method based on args passed in
	protected static final List<String> SVC_MIXED = Arrays.asList("system", "init");
	protected static final List<String> SVC_ENDPOINT_GET = Arrays.asList("ping", "list", "get", "schema", "health", "next");
	public static String method(String[] args) throws Exception {
		if(args.length < 2) {
			throw new Exception("No method provided. Please consult https://docs.minusonedb.com for guidance.");
		}
		String[] parts = args[1].split("/");
		String end = parts[parts.length - 1];
		if(SVC_MIXED.contains(parts[0])) {
			return args.length > 2 ? Http.POST : Http.GET;
		} else if(SVC_ENDPOINT_GET.contains(end)) {
			return Http.GET;
		} else {
			return Http.POST;
		}
	}
	protected static Map<String, Object> params(String[] args) throws Exception {
		Map<String, Object> params = new HashMap<>();
		for(var i = 0; i < args.length; i = i + 2) {
			if(args[i].indexOf('-') != 0) {
				throw new Exception("Looks like you are missing a parameter argument. Parameters must start with '-'.");
			}
			if(args.length < i + 2) {
				throw new Exception("Looks like you are missing a parameter value. Odd number of arguments provided.");
			}
			params.put(args[i].substring(1), args[i + 1]);
		}
		return params;
	}

}
