/********************************************************************************************************************************
 * Copyright 2021-2023 MinusOne, Inc.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom
 * the Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR
 * IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 ********************************************************************************************************************************/
window.m1events = function(config) {
	"use strict";

	const SESSION_PARAMETER = "_s";

	const INT_WRITE_DEFAULT = 1000;
	const INT_UPDATE_DEFAULT = 1000;

	var loadTime = Date.now();
	var token;

	/********************************************************************
	 * Interface
	 */

	// initialize and assign id
	this.sessionStart = sessionStart;
	this.sessionId = sessionId;
	this.auth = auth;

	// methods for sending data
	this.write = write;
	this.updateSession = updateSession;
	this.deleteSession = deleteSession;
	this.flush = flush;

	// helpers for instrumenting html elements
	this.itemEvent = itemEvent;
	this.navComponents = navComponents;
	this.urlFromComponents = urlFromComponents;
	this.canonicalizeUrl = canonicalizeUrl;
	this.setParameters = setParameters;
	this.removeParameters = removeParameters;

	// configuration
	this.name = name;
	this.sessionParameter = sessionParameter;
	this.forceNewSession = forceNewSession;
	this.sessionStore = sessionStore;
	this.writeInterval = writeInterval;
	this.updateInterval = updateInterval;
	this.publish = publish;
	this.async = async;
	this.parameter = parameter;
	this.parameterObject = parameterObject;
	
	// miscellaneous utilities
	this.matchesAny = matchesAny;

	/********************************************************************
	 * Implementation
	 */

	window.m1 = window.m1||{};

	// Unique session, random id and start time
	var session;
	// persistent messages ready to be sent
	var queue = [];
	// messages for the session store
	var updates = [];

	function sessionStart() {
		session = {
			"id": newSessionId(),
			"start": Date.now()
		};
	}
	function sessionId() {
		return session ? session.id : null;
	}
	function write(base) {
		queue.push(event(base));
	}
	function updateSession(base) {
		updates.push({ "upsert": [event(base)] });
	}
	function deleteSession() {
		updates.push({ "delete": [session.id] });
		session = null;
	}
	function itemEvent(item, type) {
		var out = { "event": itemData(item) };
		if(type) {
			out.event.type = type;
		}
		return out;
	}
	function name(name) {
		if (name) {
			config.name = name;
		}
		return config.name;
	}
	function sessionParameter(param) {
		if (param) {
			config.sessionParameter = param;
		}
		return config.sessionParameter || SESSION_PARAMETER;
	}
	function forceNewSession(param) {
		if (param) {
			config.forceNewSession = param;
		}
		return config.forceNewSession || false;
	}
	function sessionStore(store) {
		if (store) {
			config.session = store;
		}
		return config.session;
	}
	function writeInterval(int) {
		if (int != null) {
			config.writeInterval = int;
		}
		return config.writeInterval || INT_WRITE_DEFAULT;
	}
	function updateInterval(int) {
		if (int != null) {
			config.updateInterval = int;
		}
		return config.updateInterval || INT_UPDATE_DEFAULT;
	}
	function publish(publish) {
		if (publish != null) {
			config.publish = publish;
		}
		return config.publish != null ? config.publish : true;
	}
	function async(async) {
		if (async != null) {
			config.async = async;
		}
		return config.async != null ? config.async : false;
	}
	function nestedProperty(e, props) {
		if(!props || !props.length) {
			return e;
		} else if(e == null) {
			return null;
		} else {
			return nestedProperty(e[props[0]], props.slice(1));
		}
	}
	function ids(events, props) {
		return events.map(function(event, i) {
			return nestedProperty(event, props);
		});
	}
	function removeEvents(list, removed, props) {
		var done = ids(removed, props);
		return list.filter(function(value) {
			return !done.includes(nestedProperty(value, props));
		});
	}

	/********************************************************************
	 * Internals
	 */
	const SUFFIX_DOMAIN = ".minusonedb.com/";
	const SVC_AUTH = "auth";
	const SVC_WRITE = "write";
	const SVC_SESSION_UPDATE = "session/update";
	
	function endpoint() {
		return config.name == "dev" ? "http://localhost:8080/svc/" : "https://" + config.name + SUFFIX_DOMAIN;
	}

	function auth() {
		return fetch(endpoint() + SVC_AUTH, {
			method: "POST",
			body: new URLSearchParams({
				username: config.username,
				password: config.password
			})
		}).then(function(ar) {
			return ar.text().then(function(t) {
				return token = t;
			});
		});
	}

	function authPost(url, data, failCallback) {
		var p;
		if(!config.disabled) {
			// NOTE: catch() is only called on network errors and whatnot
			// We ignore on the assumption that posts will be retried
			/*
			p = $.ajax({
				method: "POST",
				url: url,
				headers: {
					"m1-auth-token": token
				},
				data: data
			});
			 */
			p = fetch(url, {
				method: "POST",
				headers: {
					"m1-auth-token": token
				},
				body: new URLSearchParams(data)
			}).then(function(response) {
				if(response.status == 401) {
					auth().then(function(t) {
						p = authPost(url, data, failCallback);
					});
				} else if(response.status != 200) {
					failCallback(response);
				}
				return response;
			}).catch(failCallback);
		}
		return p;
	}
	function postEvents() {
		var q = queue;
		if (q && q.length > 0) {
			queue = removeEvents(queue, q, ["id"]);
			authPost(endpoint() + SVC_WRITE, {
				"publish": publish(),
				"async": async(),
				"items": JSON.stringify(q)
			}, function() {
				queue = queue.concat(q);
			});
		}
	}
	function postUpdates() {
		var q = updates;
		if (sessionStore() && q && q.length > 0) {
			updates = removeEvents(updates, q, ["upsert", 0, "id"]);
			authPost(endpoint() + SVC_SESSION_UPDATE, {
				"store": sessionStore(),
				"ops": JSON.stringify(q)
			}, function(data) {
				// NOTE: this works because we add a single event per upsert. 
				// If that were not the case removeEvents() would need to be generalized.
				updates = updates.concat(q);
			});
		}
	}
	function round(number) {
		return number ? Math.floor(number) : null;
	}
	// NOTE: flush is destructor-ish; currently only meant to be used when navigating away from page 
	function flush() {
		clearTimeout(eventTimer);
		clearTimeout(updateTimer);
		if(queue && queue.length > 0) {		
			var data = new FormData();
			data.append("publish", publish());
			data.append("async", async());
			data.append("items", JSON.stringify(queue));

			if(!config.disabled) {
				fetch(endpoint() + SVC_WRITE, {
					"method": "POST",
					"body" : data,
					"keepalive" : true,
					"mode" : "cors",
					"headers" : {
						"m1-auth-token" : token
					}
				});
			}
		}
		queue = [];
		if(updates && updates.length > 0 && !config.disabled) {
		 	fetch(endpoint() + SVC_SESSION_UPDATE, {
		 		"method": "POST",
		 		"body" : formData({ "store" : sessionStore(),
														"ops"   : JSON.stringify(updates) }),
		 		"keepalive" : true,
		 		"mode" : "cors",
		 		"headers" : {
		 			"m1-auth-token" : token
		 		}
		 	});
		}
		updates = [];
		// TODO this is necessary for firefox (it doesn't respect keepalive)
		const time = Date.now();
		while ((Date.now() - time) < 250) { 
		}
	}
	
	var eventTimer = setInterval(postEvents, writeInterval());
	var updateTimer = setInterval(postUpdates, updateInterval());

	function event(base) {
		var now = Date.now();
		if (!session) {
			sessionStart();
		}
		return extend({
			"session": session,
			"id": uuid(),
			"time": now,
			"online": navigator.onLine,
			"page": extend(true, navComponents(window.location.href), {
				"title": document.title,
				"hasFocus": document.hasFocus(),
				"timeOnPage": loadTime ? round(now - loadTime) : -1,
				"scrollTop": round(document.documentElement.scrollTop),
				"scrollLeft": round(document.documentElement.scrollLeft),
				"mouseX": round(mouseX),
				"mouseY": round(mouseY),
				"mouseTime": round(mouseTime)
			}),
			"referrer": navComponents(document.referrer),
			"settings": {
				"language": navigator.language,
				"cookies": navigator.cookieEnabled,
				"doNotTrack": (window.doNotTrack || navigator.doNotTrack || navigator.msDoNotTrack) ? true : false,
			},
			"system": {
				"userAgent": navigator.userAgent,
				"platform": navigator.platform,
				"os": navigator.oscpu,
				"maxTouchPoints" : round(navigator.maxTouchPoints)
			},
			"screen": {
				"height": round(screen.height),
				"width": round(screen.width),
				"availableHeight": round(screen.availHeight),
				"availableWidth": round(screen.availWidth),
				"colorDepth": round(screen.colorDepth),
				"orientation": (screen.orientation || {}).type || screen.mozOrientation || screen.msOrientation,
				"angle": round((screen.orientation || {}).angle)
			},
			"window": {
				"innerHeight": round(window.innerHeight),
				"outerHeight": round(window.outerHeight),
				"innerWidth": round(window.innerWidth),
				"outerWidth": round(window.outerWidth),
				"x": round(window.screenX),
				"y": round(window.screenY)
			}
		}, base || {});
	}

	var mouseX, mouseY, mouseTime;
	document.addEventListener('mousemove', function(event) {
		mouseX = event.clientX;
		mouseY = event.clientY;
		mouseTime = round(event.timeStamp);
	});

	function newSessionId() {
		return forceNewSession() ? uuid() : parameter(sessionParameter()) || uuid();
	}
	function parameter(param) {
		var val = new URLSearchParams(window.location.search).get(param);
		if(!val) {
			var elt = document.querySelector("input[type='hidden'][name='" + param + "']");
			val = elt ? elt.value : null;
		}
		return val;
	}
	function parameterObject(param) {
		var out = {};
		document.querySelectorAll("input[type='hidden'][name^='" + param + "." +"']").forEach(function(elt, i) {
			out = setProperty(out, elt.getAttribute("name").substring(param.length+1), elt.value);
		});
		var params = new URLSearchParams(window.location.search);
		for(var key of params.keys()) {
			if(key.startsWith(param + ".")) {
				out = setProperty(out, key.substring(param.length+1), params.get(key));
			}
		}
		return out;
	}
	function setProperty(out, prop, value) {
		return setPropertyParts(out, prop.split("."), value);
	}
	function setPropertyParts(out, parts, value) {
		if(parts.length == 1) {
			out[parts[0]] = value;
			return out;
		} else {
			out[parts[0]] = setPropertyParts(out[parts[0]]||{}, parts.slice(1), value);
			return out;
		}
	}
	function uuid() {
		return ([1e7] + -1e3 + -4e3 + -8e3 + -1e11).replace(/[018]/g, 
			c => (c ^ crypto.getRandomValues(new Uint8Array(1))[0] & 15 >> c / 4).toString(16));
	}
	function setParameters(urlOrFragment, params) {
		var components = navComponents(urlOrFragment);
		if (!components.params) {
			components.params = {};
		}
		for (var param of Object.keys(params)) {
			components.params[param] = params[param];
		}
		return urlFromComponents(components);
	}
	function removeParameters(urlOrFragment, params) {
		var components = navComponents(urlOrFragment);
		if (components.params) {
			for (var param of params) {
				delete components.params[param];
			}
		}
		return urlFromComponents(components);
	}
	function getParameters(paramString) {
		var params = {};
		var parts = paramString.split("&");
		for (var i = 0; i < parts.length; i++) {
			if (parts[i] != "") {
				var keyVal = parts[i].split("=");
				params[keyVal[0]] = params[keyVal[0]] || [].concat([keyVal[1]]);
			}
		}
		return params;
	}
 	function navComponents(url) {
		if(url != null && url != "")  {
			if (url.startsWith("mailto:")) {
				var email = url.substr(url.indexOf("mailto:") + "mailto:".length);
				var domain = email.indexOf("@") > -1 ? email.substr(email.indexOf("@") + 1) : null;
				return {
					"url" : url,
					"protocol" : "mailto",
					"domain" : domain,
					"primaryDomain" : primaryDomain(domain),
					"path" : email
				};
			} else {
				var protoParts = url.split("://");
				var protoIndex = url.indexOf("://");
				var protocol = protoIndex > -1 ? url.substring(0, protoIndex) : null;
				var page = url.substring(protoIndex == -1 ? 0 : protoIndex + 3);
				var urlParts = page.split("#");
				var pathParam = urlParts[0].split("?");
				var path = pathParam[0];
				var baseParts = (path.indexOf("/") > -1 ? path.split("/")[0] : path).split(":");
				var domain = protoIndex > -1 ? baseParts[0] : null;
				var components = {
					"url": url,
					"protocol": protocol,
					"hash": (urlParts.length > 1 ? urlParts[1] : null),
					//"params": (pathParam.length > 1 ? getParameters(pathParam[1]) : null),
					"domain": domain,
					"primaryDomain" : primaryDomain(domain),
					"port": baseParts.length > 1 ? round(baseParts[1]) : null,
					"path": (protoIndex > -1) ? (path.indexOf("/") > -1 ? path.substr(path.indexOf("/")) : "/") : (path.startsWith("/") ? path : "/" + path)
				};
				if(pathParam.length > 1) {
					components.params = getParameters(pathParam[1]);
				}
				return components;
			}
		} else {
			return null;
		}
	}
	function paramString(params) {
		var result = "";
		if (params && Object.keys(params).length > 0) {
			result += "?";
			for(var param of Object.keys(params)) {
				if (Array.isArray(params[param])) {
					for (var val of params[param]) {
						result += param + "=" + val + "&";
					}
				} else {
					result += param + "=" + params[param] + "&";
				}
			}
		} 
		return result;
	}
	function urlFromComponents(components) {
		var base = (components.protocol) ? components.protocol + "://" + components.domain + (components.port ? ":" + components.port : "") : "";
		return base + components.path + paramString(components.params) + (components.hash ? "#" + components.hash : "");
	}
	function canonicalizeUrl(urlOrFragment) {
		var urlComponents = navComponents(urlOrFragment);
		if(!urlComponents.protocol) {
			var currentComponents = navComponents(window.location.href);
			urlComponents.protocol = currentComponents.protocol;
			urlComponents.domain = currentComponents.domain;
			urlComponents.port = currentComponents.port;
			return urlFromComponents(urlComponents);
		} else {
			return urlOrFragment;
		}
	}
	function primaryDomain(domain) {
		if (domain) {
			var domainParts = domain.split("."); 
			return domainParts.length > 1 ? domainParts[domainParts.length-2] + "." + domainParts[domainParts.length-1] : null;
		} else {
			return null;
		}
	}
	function itemData(item) {
		var props = { element: item.nodeName };
		for(var i=0;i<item.attributes.length;i++) {
			var a = item.attributes[i];
			if(a.name == "href" || a.name == "data-href") {
				props["href"] = navComponents(canonicalizeUrl(a.value));
			} else if(a.name.startsWith("data-")) {
				props[a.name.substr(5)] = a.value;
			}
		}
		return extend(props, formParams(item));
	}
	function formParams(form) {
		var params = {};
		if(form.nodeName.toLowerCase() == "form") {
			form.querySelectorAll("input, select, textarea").forEach(function(item) {
				var name = item.getAttribute("name");
				if(item.getAttribute("data-private") != "true" && name) {
					params[name] = item.value;
				}
			});
		}
		return params;
	}
	function formData(formParams) {
		var data = new FormData();
		for(var param of Object.keys(formParams)) {
			data.append(param, formParams[param]);
		}
		return data;
	}
	function matchesAny(s, regexes) {
		for (var regex of regexes) {
			if (regex.test(s)) {
				return true;
			}
		}
		return false;
	}
	// TODO: below really deserves to be pulled in from a library somewhere
	// NOTE: pretty much exists to avoid a dependency on jquery
	var extend = window.m1.extend = function() {
		var params = array(arguments);
		var deep = false;
		if(params[0] === true) {
			deep = params.shift();
		}
		if(!arguments || arguments.length == 0) {
			return null;
		}
		var out = params.shift();
		params.forEach(function(map) {
			out = merge(out, map, deep);
		});
    return out;
	}
	function merge(base, merged, deep) {
		// degenerate case that may come up in some deep merges
		if(typeof base != "object") {
			return merged;
		}
    for(var key in merged) {
			if(merged.hasOwnProperty(key)) {
				if(deep && (typeof merged[key] == "object")) {
					base[key] = merge(base[key], merged[key], deep);
				} else {
					base[key] = merged[key];
				}
			}
		}
		return base;
	}
	// mainly used as a hack to avoid using arguments (or ES6 spread)
	function array(arrayLike) {
		var out = [];
		for(var i=0;i<arrayLike.length;i++) {
			out[i] = arrayLike[i];
		}
		return out;
	}
};
