/********************************************************************************************************************************
 * Copyright 2021-2023 MinusOne, Inc.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom
 * the Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR
 * IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 ********************************************************************************************************************************/


/* EVENT_PARAMS

Will capture string parameters from the app to append to events. 

The framework will look at javascript objects that can be embedded 
directly in the page server-side, or created client-side.
It will also look for hidden variables with matching names.

Ex:
var campaign = "1777";

<input type="hidden" name="campaign" value="1777" />

Useful for campaign, source, etc.

*/
const EVENT_PARAMS = ["campaign"];


/* APP_OBJECTS

Objects can be embedded in events by adding the name of the object
to APP_OBJECTS. Parameters are searched as above but url parameters
will also be included.

Ex:
var user = { id: "123", username: "johndoe" }

<input type="hidden" name="user.id" value="123" />
<input type="hidden" name="user.username" value="johndoe" />

...url?user.id=123&user.username=johndoe

Similarly cookie objects can track your anonymous users across sessions.

*/
const APP_OBJECTS = ["user", "cookie"];

(function() {
	"use strict";
	/// For more information about configuration and settings please see the Event Capture guide at https://docs.minusonedb.com

	/*
		The sessionMetada object (and associated code below) is an example
		of how additional data can be appended to events as needed. Most 
		use cases will want this or something like it.
	*/
	// Your html can embed extensions to this variable with examples such as
	// window.sessionMetadata.user = {...};
	window.sessionMetadata = {
	};
	function sessionMetadata(events) {
		for(var o of APP_OBJECTS) {
			window.sessionMetadata[o] = events.parameterObject(o);
		}
		for(var p of EVENT_PARAMS) {
			window.sessionMetadata[p] = events.parameter(p);
		}
		return window.sessionMetadata;
	}
	function getAppEvent(meta, event) {
		return event ? m1.extend({}, meta, event) : meta;
	}

	document.addEventListener("DOMContentLoaded", function() {
		var events = new m1events({

			///////////////////////////////////////////////////////////////////
			// REQUIRED PARAMETERS

			/* Your minusonedb environment is available at
			   https://<env>-<account>.minusonedb.com */
			name: "<env>-<account>",

			/* NOTE: this should be a write-only system account, typically only
			   used for instrumentation. */
			username: "<username>",

			/* NOTE: this password is insecure. System accounts for instrumentation
			   are typically not secured. */
			password: "<password>",

			/* Optional value to specify parameter name to be used for a hidden input
			   or querystring parameter for pre-assigned session ids */
			// sessionParameter: "<param>",

			/* Set to false for development to get helpful error messages;
			   set to true for production to maximize performance
			    NOTE: for development you will also likely want to set
			    publish-permissive to false in the associated m1db environment. */
			async: false, 

			/* Frequency (in ms) with which events are posted to the minusonedb
			   /write endpoint. */
			// writeInterval: 1000,

			/* If true, events will be published to the data lake and any live stores
			   for near real-time querying; otherwise events will only be written to
			   the archive layer. */
			publish: true


			///////////////////////////////////////////////////////////////////
			// SESSION STORE-SPECIFIC PROPERTIES
			
			/* Name of session store; only relevant if you're using a session store to
			   manage heartbeats (not generally recommended). */
			// session: "stash",

			/* Frequency (in ms) with which heartbeats are sent to the session store;
			only relevant if you're using a session store to manage heartbeats. */
			// updateInterval: 1000, 
		});

		var forceNewSession = performance.getEntriesByType("navigation")[0].type != "back_forward" &&
				performance.getEntriesByType("navigation")[0].type != "reload" &&
				!(window.location.href &&
					document.referrer &&
					events.navComponents(window.location.href)["primaryDomain"] == events.navComponents(document.referrer)["primaryDomain"]);
		events.forceNewSession(forceNewSession);
	
		window.addEventListener("beforeunload", function(e) { 
			// Used to ensure sessions are preserved across back/forward navigations
			history.replaceState(null, "", events.setParameters(window.location.href, {[events.sessionParameter()] : events.sessionId()}));
		 	events.flush();
		});
		
		var meta = sessionMetadata(events, APP_OBJECTS);
		
		/* Single-page applications will typically want a view event on hash changes;
		   server-side html applications may want a different type of event. */
		window.addEventListener("hashchange", function(e) {
			events.write(getAppEvent(meta, { "event": { "type": "view" }}));
		});
		
		/* Server-side html applications typically want a "pageLoad" event that may include url params.
		   Single-page applications will typically consider this a "view" instead. */
		//events.write(getAppEvent(meta, {"event": { "type": "pageLoad" } }));
		events.write(getAppEvent(meta, { "event": { "type": "view" }}));


		/* Initialize a session if needed. */
		if (forceNewSession || new URLSearchParams(window.location.search).get(events.sessionParameter()) == null) {
			events.write(getAppEvent(meta, { "event": { "type": "sessionStart" }}));
		}
		history.replaceState(null, "", events.removeParameters(window.location.href, [events.sessionParameter()]));
		
		/* This can be wired up to any navigation actions in a single-page application using an
		   appropriate selector in place of "a". */
		document.querySelectorAll("a").forEach(function(a) {
			a.addEventListener("click", function(e) {
				events.write(getAppEvent(meta, {
					"event": { "type": "click",
			 		"href" : events.navComponents(events.canonicalizeUrl(this.getAttribute("href")))}}));
			});
		});

		/* Used to preserve sessions as users navigate across site subdomains. */
		var parent = events.navComponents(window.location.href)["primaryDomain"];
		document.querySelectorAll("a[href*='" + parent + "']").forEach(function(a, i) {
			/* NOTE: you can add conditions here to select the subset of anchors
			   that should be updated to preserve the same session. */
			if(true) {
				a.setAttribute("href", events.setParameters(a.getAttribute("href"), {[events.sessionParameter()] : events.sessionId()}));
			}
		});
		
		/* Posts all forms and parameters. Note that inputs and selects will be
		   omitted if marked data-private="true". */
		document.querySelectorAll("form").forEach(function(form, i) {
			form.addEventListener("submit", function(e) {
				 events.write(getAppEvent(meta, events.itemEvent(this)));
			});
		});

		/////////////////////////////////////////////////////////////////////////
		// You can ignore everything below if you're uninterested in heartbeats

		const HEARTBEAT_INTERVAL_MS = 5000;	

		/* PERSISTED HEARTBEATS

		   If you want to save all heartbeats, uncomment the line below after
		   configuring your desired heartbeat interval. */
		// setInterval(function() { events.write(getAppEvent(meta, {"event": {"type":"heartbeat"}})); }, HEARTBEAT_INTERVAL_MS); 

		/* SESSION STORE HEARTBEATS
		   Using a session store with heartbeats:
		   1) Specify the name of the session store you are using above; the default is stash.
		   2) Set the desired heartbeat interval by setting the HEARTBEAT_INTERVAL_MS constant above.
		   3) Uncomment the updateSession line below to send heartbeats to your specified session store.
		   4) You must set up a process to remove heartbeats from the session store; 
			    see flusher.py in this repository for a sample.
		*/
		//setInterval(function() { events.updateSession(getAppEvent(meta, {"event": {"type":"heartbeat"}})); }, HEARTBEAT_INTERVAL_MS);		
	});
})();
