# Copyright 2021-2022 MinusOne, Inc.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation
# files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,
# modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom
# the Software is furnished to do so, subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR
# IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

import requests
import time
import json

# TODO this file and/or this class probably needs to be renamed m1db to mimic our desired naming conventions around client libraries

class M1Client:
	def __init__(self, server, username, password, ip):
		self.server = server
		self.username = username
		self.password = password
		self.ip = ip
		self.backoffInterval = 4
		self.token = None
		self.setAuthToken() # TODO it might be clever to not retry forever in this case and immediately fail, because it's likely there's a configuration issue

	# TODO parameterize # of retries; we may not want to retry forever for certain ops endpoints?
	
	def responseOkay(self, resp, path, data):
		# TODO we need a different error code here to indicate a problem with publish rather than using this heuristic
		if path == "/publish":
			content = resp.content.decode("utf-8").strip()
			lines = [x for x in content.split("\n")]
			okay = lines[-1].startswith("Saved")
			return okay
		elif path in ("/query", "/session/query"):
			return resp.status_code // 100 == 2 # TODO This is not correct for these streaming endpoints
		else:
			return resp.status_code // 100 == 2

	def setAuthToken(self):
		if self.username is None:
			return
		self.token = self.postRetryForever("/auth", {"username" : self.username, "password" : self.password, "ip" : self.ip})

	def getRetryForever(self, path):
		return self.requestForever("GET", path, None)
		
	def postRetryForever(self, path, data):
		return self.requestForever("POST", path, data)

	def requestForever(self, requestType, path, data):
		if requestType == "GET" and data is not None:
			raise Exception("Trying to post data with GET")
		elif requestType not in ("GET", "POST"):
			raise Exception(f"Unknown request type: {requestType}")
		while True:
			try:
				headers = {} if self.token is None else { "m1-auth-token" : self.token}
				if requestType == "POST":
					resp = requests.post(f"{self.server}{path}", data, headers = headers)
				elif requestType == "GET":
					resp = requests.get(f"{self.server}{path}", headers = headers)
					
				if resp.status_code == 401:
					print(f"Problem with {self.server}{path}: code: {resp.status_code} token: {self.token} {resp.content}")
					self.setAuthToken()
				elif self.responseOkay(resp, path, data):
					return resp.content
				else:
					print(f"Problem with {self.server}{path}: code: {resp.status_code} trace:\n{self.cleanTrace(resp.content)}")
			except Exception as e:
				print(f"Exception with {path}: {e}")
			time.sleep(self.backoffInterval)
			
	def cleanTrace(self, bytes):
		return bytes.decode("utf-8").replace("\\n", "\n")
