# Copyright 2021-2022 MinusOne, Inc.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation
# files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,
# modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom
# the Software is furnished to do so, subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR
# IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
import time
import json
import uuid
import sys
from datetime import datetime

from m1 import M1Client

M1_KEY = "_m1"
CHECK_INTERVAL = 2

CONFIG_TIMEOUT_KEY = "timeoutSeconds"
CONFIG_SERVER_KEY = "server"
CONFIG_STORE_KEY = "session"
CONFIG_USERNAME_KEY = "username"
CONFIG_PASSWORD_KEY = "password"
CONFIG_IP_KEY = "ip"

def prettyTimestamp(epoch):
	return datetime.fromtimestamp(epoch/1000 if epoch else time.time()).strftime("%Y-%m-%dT%H:%M:%S")

def getTimedOutSessions():
	timedOutTime = (time.time()- int(CONFIG[CONFIG_TIMEOUT_KEY])) * 1000
	query = f"_m1.receivedUTC:[* TO {timedOutTime}]"
	print(f"session store query: {query}; looking for sessions older than {prettyTimestamp(timedOutTime)}")
	result = json.loads(CLIENT.postRetryForever("/session/query",
																							{"store" : CONFIG[CONFIG_STORE_KEY],
																							 "q" : query}))
	return result["items"]

def transform(sessions):
	for session in sessions:
		if "user" in session and "displayName" in session["user"]:
			displayName = session["user"]["displayName"]
		else:
			displayName = "a user"

		if "time" in session:
			eventTime = prettyTimestamp(session['time'])
		else:
			eventTime = "unknown"
		print(f"Preparing to flush {displayName}'s session ({session['session']['id']}); event time {eventTime} and sesson store time of {prettyTimestamp(session['_m1']['receivedUTC'])}")

		if "event" in session:
			session["event"]["type"] = "sessionTimeout"
			session["time"] = int(time.time() * 1000)
		# Insert any other modifications here between session store heartbeat entity and an event you want to materialize
		# you may want to alternatively or additionally save the old event time
	return sessions

def archiveAndDelete(sessions):
	archiveSessions(sessions) 
	deleteSessions(sessions)
		
def archiveSessions(sessions):
	CLIENT.postRetryForever("/write", {"items" : json.dumps(sessions), "publish" : True })
	
def deleteSessions(sessions):
	deleteQuery = [{"delete" : [x["session"]["id"] for x in sessions]}]
	CLIENT.postRetryForever("/session/update", { "store" : CONFIG[CONFIG_STORE_KEY], "ops" : json.dumps(deleteQuery)})

def flush():
	while True:
		sessions = getTimedOutSessions()
		if sessions:
			print(f"Flushing {len(sessions)} at {prettyTimestamp(None)}")
			archiveAndDelete(transform(sessions))
		else:
			print("nothing to flush")
		time.sleep(CHECK_INTERVAL)

with open("default.json" if len(sys.argv) == 1 else sys.argv[1]) as f:
	CONFIG = json.load(f)
	CLIENT = M1Client(CONFIG[CONFIG_SERVER_KEY], CONFIG[CONFIG_USERNAME_KEY], CONFIG[CONFIG_PASSWORD_KEY], CONFIG.get(CONFIG_IP_KEY))

flush()
